"use client"

import { useState, useEffect } from "react"
import Link from "next/link"
import Image from "next/image"
import { Check, Star, ArrowRight } from "lucide-react"
import AddToCartButton from "./AddToCartButton"

export default function FeaturedProduct() {
	const [isVisible, setIsVisible] = useState(false)

	useEffect(() => {
		const observer = new IntersectionObserver(
			([entry]) => {
				if (entry.isIntersecting) {
					setIsVisible(true)
				}
			},
			{ threshold: 0.1 },
		)

		const element = document.getElementById("featured-product")
		if (element) {
			observer.observe(element)
		}

		return () => {
			if (element) {
				observer.unobserve(element)
			}
		}
	}, [])

	const featuredProduct = {
		id: 1,
		name: "Diazepam 10mg",
		slug: "diazepam-10mg-zonder-recept",
		price: 70,
		originalPrice: 85,
		category: "Slaappillen",
		categories: ["Medicijnen", "Slaappillen", "Benzodiazepinen"],
		stock: 120,
		image: "/images/products/diazepam-10mg.jpg",
		rating: 4.9,
		reviews: 124,
		description:
			"Diazepam 10mg is een effectief medicijn dat behoort tot de benzodiazepinen. Het wordt gebruikt voor de behandeling van angst, spierspasmes, en slaapstoornissen. Het werkt door de activiteit van bepaalde natuurlijke stoffen in de hersenen te verhogen.",
		benefits: [
			"Effectieve behandeling van angst en spanning",
			"Helpt bij slaapstoornissen",
			"Verlicht spierspasmes en krampen",
			"Snelwerkend en betrouwbaar",
			"Hoogste kwaliteit gegarandeerd",
		],
	}

	const discountPercentage = Math.round(
		((featuredProduct.originalPrice - featuredProduct.price) / featuredProduct.originalPrice) * 100,
	)

	return (
		<section
			id="featured-product"
			className="py-16 bg-gradient-to-br from-[#f0f9ff] to-[#e6f7ff] relative overflow-hidden"
		>
			{/* Background elements */}
			<div className="absolute top-0 left-0 w-full h-full overflow-hidden">
				<div className="absolute top-20 right-20 w-64 h-64 rounded-full bg-[#1eaaff] opacity-10 blur-[100px]"></div>
				<div className="absolute bottom-10 left-10 w-96 h-96 rounded-full bg-[#00e889] opacity-10 blur-[120px]"></div>
				<div className="absolute -top-20 -left-20 w-72 h-72 rounded-full border border-[#1eaaff] opacity-20"></div>
			</div>

			<div className="container mx-auto px-4 relative z-10">
				<div className="text-center mb-12">
					<h2 className="text-3xl font-bold mb-4">
						Uitgelicht{" "}
						<span className="bg-clip-text text-transparent bg-gradient-to-r from-[#1eaaff] to-[#0071e6]">product</span>
					</h2>
					<p className="text-gray-600 max-w-2xl mx-auto">
						Ontdek ons meest populaire product, geliefd door duizenden klanten
					</p>
				</div>

				<div
					className={`bg-white rounded-2xl shadow-xl overflow-hidden border border-gray-100 ${isVisible ? "animate-fade-in" : "opacity-0"}`}
				>
					<div className="grid grid-cols-1 lg:grid-cols-2">
						<div className="p-8 lg:p-12 flex flex-col justify-center">
							<div className="flex items-center mb-4">
								<span className="bg-[#1eaaff]/10 text-[#1eaaff] text-sm font-medium px-3 py-1 rounded-full">
									Bestseller
								</span>
								{featuredProduct.originalPrice && (
									<span className="ml-3 bg-[#ff6b6b]/10 text-[#ff6b6b] text-sm font-medium px-3 py-1 rounded-full">
										{discountPercentage}% Korting
									</span>
								)}
							</div>

							<h3 className="text-3xl font-bold text-gray-900 mb-4">{featuredProduct.name}</h3>

							<div className="flex items-center mb-6">
								<div className="flex items-center text-yellow-400 mr-3">
									{[...Array(5)].map((_, i) => (
										<Star
											key={i}
											size={18}
											className={i < Math.floor(featuredProduct.rating) ? "fill-yellow-400" : "text-gray-300"}
										/>
									))}
								</div>
								<span className="text-gray-600">
									{featuredProduct.rating} ({featuredProduct.reviews} reviews)
								</span>
							</div>

							<p className="text-gray-600 mb-6">{featuredProduct.description}</p>

							<div className="mb-8">
								<h4 className="font-medium text-gray-900 mb-3">Voordelen:</h4>
								<ul className="space-y-2">
									{featuredProduct.benefits.map((benefit, index) => (
										<li key={index} className="flex items-start">
											<Check size={18} className="text-[#00e889] mr-2 mt-0.5" />
											<span className="text-gray-600">{benefit}</span>
										</li>
									))}
								</ul>
							</div>

							<div className="flex items-end mb-6">
								{featuredProduct.originalPrice && (
									<span className="text-lg text-gray-500 line-through mr-3">
										€{featuredProduct.originalPrice.toFixed(2)}
									</span>
								)}
								<span className="text-3xl font-bold text-[#1eaaff]">€{featuredProduct.price.toFixed(2)}</span>
							</div>

							<div className="flex flex-col sm:flex-row gap-4">
								<AddToCartButton
									id={featuredProduct.id}
									name={featuredProduct.name}
									price={featuredProduct.price}
									image={featuredProduct.image}
									className="py-3 px-6 bg-gradient-to-r from-[#1eaaff] to-[#0071e6] hover:from-[#0a8aff] hover:to-[#0060c9] text-white font-medium rounded-full transition-all duration-300 shadow-md hover:shadow-lg flex-1"
								/>

								<Link href={`/product/${featuredProduct.slug.toLowerCase().split(" ").join("-")}`} className="group">
									<div className="py-3 px-6 bg-white border border-gray-300 hover:border-[#1eaaff] text-gray-800 hover:text-[#1eaaff] font-medium rounded-full flex items-center justify-center transition-all duration-300 flex-1">
										<span>Meer informatie</span>
										<ArrowRight
											size={18}
											className="ml-2 transform group-hover:translate-x-1 transition-transform duration-300"
										/>
									</div>
								</Link>
							</div>
						</div>

						<div className="bg-[#f8fafc] p-8 lg:p-0 flex items-center justify-center relative">
							<div className="absolute top-0 left-0 w-full h-full overflow-hidden opacity-50">
								<div className="absolute top-1/4 left-1/4 w-32 h-32 rounded-full bg-[#1eaaff] blur-[80px]"></div>
								<div className="absolute bottom-1/4 right-1/4 w-40 h-40 rounded-full bg-[#00e889] blur-[100px]"></div>
							</div>

							<div className="relative w-full h-[300px] lg:h-[400px] animate-float">
								<Image
									src={featuredProduct.image || "/placeholder.svg"}
									alt={featuredProduct.name}
									fill
									className="object-contain p-8"
								/>
							</div>
						</div>
					</div>
				</div>
			</div>
		</section>
	)
}
