"use client"

import { useState, useEffect } from "react"
import Image from "next/image"
import { Shield, Truck, CreditCard, Clock } from "lucide-react"

export default function InfoSection() {
  const [isVisible, setIsVisible] = useState(false)

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.1 },
    )

    const element = document.getElementById("info-section")
    if (element) {
      observer.observe(element)
    }

    return () => {
      if (element) {
        observer.unobserve(element)
      }
    }
  }, [])

  const infoPoints = [
    {
      icon: <Shield className="h-10 w-10 text-[#1eaaff]" />,
      title: "Veilig & Betrouwbaar",
      description:
        "Alle medicijnen worden zorgvuldig gecontroleerd op kwaliteit en veiligheid voordat ze worden verzonden.",
    },
    {
      icon: <Truck className="h-10 w-10 text-[#1eaaff]" />,
      title: "Discrete Verzending",
      description: "Uw bestelling wordt verzonden in neutrale verpakking zonder verwijzing naar de inhoud.",
    },
    {
      icon: <CreditCard className="h-10 w-10 text-[#1eaaff]" />,
      title: "Veilige Betaling",
      description: "Meerdere beveiligde betaalmethoden beschikbaar, inclusief iDEAL en Bancontact.",
    },
    {
      icon: <Clock className="h-10 w-10 text-[#1eaaff]" />,
      title: "Snelle Levering",
      description: "Bestellingen worden binnen 24 uur verwerkt en geleverd binnen 1-2 werkdagen.",
    },
  ]

  return (
    <section id="info-section" className="py-16 bg-white relative overflow-hidden">
      {/* Background elements */}
      <div className="absolute inset-0 bg-[url('/grid-pattern.png')] bg-center opacity-5"></div>

      <div className="container mx-auto px-4">
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 items-center">
          <div className={`${isVisible ? "animate-slide-left" : "opacity-0"}`}>
            <h2 className="text-3xl font-bold mb-6">
              Waarom{" "}
              <span className="bg-clip-text text-transparent bg-gradient-to-r from-[#1eaaff] to-[#0071e6]">
                Geenrecept
              </span>{" "}
              vertrouwen?
            </h2>

            <p className="text-gray-600 mb-8">
              Bij Geenrecept begrijpen we dat het kopen van medicijnen online een kwestie van vertrouwen is. Daarom
              streven we ernaar om de meest betrouwbare en veilige online apotheek te zijn, met een focus op kwaliteit,
              discretie en klanttevredenheid.
            </p>

            <div className="space-y-6">
              {infoPoints.map((point, index) => (
                <div
                  key={index}
                  className={`flex items-start ${isVisible ? "animate-fade-in" : "opacity-0"}`}
                  style={{ animationDelay: `${index * 0.2}s` }}
                >
                  <div className="mr-4 p-3 rounded-lg bg-[#1eaaff]/10">{point.icon}</div>
                  <div>
                    <h3 className="font-semibold text-gray-900 mb-1">{point.title}</h3>
                    <p className="text-gray-600">{point.description}</p>
                  </div>
                </div>
              ))}
            </div>
          </div>

          <div className={`relative hidden sm:block ${isVisible ? "animate-slide-right" : "opacity-0"}`}>
            <div className="relative bg-gradient-to-br from-[#f0f9ff] to-[#e6f7ff] rounded-2xl p-8 shadow-xl overflow-hidden">
              <div className="absolute top-0 left-0 w-full h-full overflow-hidden opacity-50">
                <div className="absolute top-1/4 left-1/4 w-32 h-32 rounded-full bg-[#1eaaff] blur-[80px]"></div>
                <div className="absolute bottom-1/4 right-1/4 w-40 h-40 rounded-full bg-[#00e889] blur-[100px]"></div>
              </div>

              <div className="relative z-10">


                <div className="mt-6 bg-white/80 backdrop-blur-sm p-6 rounded-xl border border-white/50">
                  <h3 className="text-xl font-bold text-gray-900 mb-2">Professionele Service</h3>
                  <p className="text-gray-600">
                    Ons team van experts staat klaar om u te helpen bij het vinden van de juiste medicijnen voor uw
                    behoeften, met persoonlijk advies en ondersteuning.
                  </p>
                </div>
              </div>

              {/* Decorative elements */}
              <div className="absolute -top-3 -left-3 w-6 h-6 bg-[#1eaaff] rounded-full"></div>
              <div className="absolute -bottom-3 -right-3 w-6 h-6 bg-[#00e889] rounded-full"></div>
            </div>
          </div>
        </div>
      </div>
    </section>
  )
}
