"use client"

import { useState, useEffect } from "react"
import Image from "next/image"
import { ChevronLeft, ChevronRight, Quote, Star } from "lucide-react"

const testimonials = [
	{
		id: 1,
		name: "Sophie de Jong",
		location: "Rotterdam",
		rating: 5,
		text: "Binnen 2 dagen mijn bestelling ontvangen! Alles discreet verpakt en precies wat ik nodig had. Geenrecept.com is betrouwbaar en snel.",
		image: "/diverse-group.png",
	},
	{
		id: 2,
		name: "Mark van Dijk",
		location: "Rotterdam",
		rating: 5,
		text: "Ik was eerst wat sceptisch, maar na mijn eerste bestelling ben ik overtuigd. Snelle levering en goede klantenservice.",
		image: "/diverse-woman-portrait.png",
	},
	{
		id: 3,
		name: "Eva Vermeer",
		location: "Utrecht",
		rating: 5,
		text: "Heel fijn dat ik geen recept nodig had. Het bestelproces was eenvoudig en duidelijk. Zeker een aanrader!",
		image: "/thoughtful-man.png",
	},
	{
		id: 4,
		name: "Tom Bakker",
		location: " Den Haag",
		rating: 5,
		text: "Top service en goede producten. Ik heb al meerdere keren besteld en ben elke keer tevreden. Ga zo door!",
		image: "/thoughtful-man.png",
	},
]

export default function TestimonialSection() {
	const [currentIndex, setCurrentIndex] = useState(0)
	const [isVisible, setIsVisible] = useState(false)
	const [autoplay, setAutoplay] = useState(true)

	const nextTestimonial = () => {
		setCurrentIndex((prev) => (prev === testimonials.length - 1 ? 0 : prev + 1))
	}

	const prevTestimonial = () => {
		setCurrentIndex((prev) => (prev === 0 ? testimonials.length - 1 : prev - 1))
	}

	useEffect(() => {
		const observer = new IntersectionObserver(
			([entry]) => {
				if (entry.isIntersecting) {
					setIsVisible(true)
				} else {
					setIsVisible(false)
				}
			},
			{ threshold: 0.1 },
		)

		const element = document.getElementById("testimonial-section")
		if (element) {
			observer.observe(element)
		}

		return () => {
			if (element) {
				observer.unobserve(element)
			}
		}
	}, [])

	useEffect(() => {
		let interval
		if (autoplay && isVisible) {
			interval = setInterval(() => {
				nextTestimonial()
			}, 5000)
		}
		return () => clearInterval(interval)
	}, [autoplay, isVisible, currentIndex])

	return (
		<section id="testimonial-section" className="py-20 relative overflow-hidden">
			{/* Background elements */}
			<div className="absolute inset-0 bg-gradient-to-br from-[#1eaaff]/5 to-[#0071e6]/5"></div>
			<div className="absolute top-0 left-0 w-full h-full overflow-hidden">
				<div className="absolute top-20 left-20 w-64 h-64 rounded-full bg-[#1eaaff] opacity-5 blur-[100px]"></div>
				<div className="absolute bottom-10 right-10 w-96 h-96 rounded-full bg-[#00e889] opacity-5 blur-[120px]"></div>
				<div className="absolute -top-20 -right-20 w-72 h-72 rounded-full border border-[#1eaaff] opacity-20"></div>
			</div>

			<div className="container mx-auto px-4 max-w-6xl relative z-10">
				<div className="text-center mb-12">
					<h2 className="text-3xl font-bold mb-4">
						Wat onze klanten{" "}
						<span className="bg-clip-text text-transparent bg-gradient-to-r from-[#1eaaff] to-[#0071e6]">zeggen</span>
					</h2>
					<p className="text-gray-600 max-w-2xl mx-auto">
						Ontdek waarom klanten ons vertrouwen voor hun medicijnbehoeften
					</p>
				</div>

				<div className="relative mx-auto max-w-4xl">
					<div className="absolute -left-4 top-1/2 -translate-y-1/2 text-[#1eaaff]/20 opacity-50">
						<Quote className="h-24 w-24" />
					</div>

					<div className="relative bg-white/50 backdrop-blur-sm rounded-2xl p-10 shadow-xl border border-white/50">
						{testimonials.map((testimonial, index) => (
							<div
								key={testimonial.id}
								className={`transition-opacity duration-500 ${index === currentIndex ? "block opacity-100" : "hidden opacity-0"
									}`}
							>
								<div className="mb-8 text-center text-lg italic leading-relaxed text-gray-700">
									"{testimonial.text}"
								</div>

								<div className="flex flex-col items-center justify-center">
									<h3 className="text-xl font-medium text-gray-900">{testimonial.name}</h3>
									<p className="mb-3 text-sm text-gray-500">{testimonial.location}</p>
									<div className="flex items-center mb-4">
										{[...Array(5)].map((_, i) => (
											<Star
												key={i}
												size={18}
												className={i < testimonial.rating ? "text-yellow-400 fill-yellow-400" : "text-gray-300"}
											/>
										))}
									</div>
								</div>
							</div>
						))}
					</div>

					<div className="mt-8 flex justify-center space-x-4">
						<button
							onClick={() => {
								prevTestimonial()
								setAutoplay(false)
							}}
							className="rounded-full bg-white p-3 shadow-md hover:bg-[#1eaaff] hover:text-white transition-colors duration-300"
							aria-label="Previous testimonial"
						>
							<ChevronLeft className="h-5 w-5" />
						</button>
						<button
							onClick={() => {
								nextTestimonial()
								setAutoplay(false)
							}}
							className="rounded-full bg-white p-3 shadow-md hover:bg-[#1eaaff] hover:text-white transition-colors duration-300"
							aria-label="Next testimonial"
						>
							<ChevronRight className="h-5 w-5" />
						</button>
					</div>
				</div>
			</div>
		</section>
	)
}
